// /api/discover/embryos — Niche Embryo Detector
// Finds sub-5k sub channels with disproportionate viral videos.
// Uses ytGet() for automatic key rotation on 403/quota errors.

import { NextResponse } from 'next/server';
import { ytGet } from '../../../../lib/youtubeApi';
import { clusterTitles } from '../../../../lib/cluster';

const SEED_QUERIES = [
    'ai tools', 'van life', 'tiny house', 'urban farming', 'side hustle',
    'quiet luxury', 'digital nomad', 'sleep hacks', 'cold plunge', 'solopreneurship',
    'notion templates', 'ai automation', 'micro saas', 'steel mace workout', 'ruck training',
];

export async function GET() {
    try {
        const embryoCandidates: { title: string; channel: string; channelId: string; views: number; subs: number; ratio: number; thumbnail: string; videoId: string }[] = [];

        // Pick 3 random seed queries (was 5 — reduces quota from 500 to 300 units)
        const selected = SEED_QUERIES.sort(() => 0.5 - Math.random()).slice(0, 3);

        for (const q of selected) {
            try {
                // Get recent videos (search.list = 100 units)
                const searchData = await ytGet('/search', {
                    q, type: 'video',
                    order: 'date', part: 'snippet',
                    maxResults: 10,
                    publishedAfter: new Date(Date.now() - 30 * 86400 * 1000).toISOString(),
                });

                const videoIds: string[] = (searchData.items || []).map((v: any) => v.id.videoId).filter(Boolean);
                const channelIds: string[] = (searchData.items || []).map((v: any) => v.snippet.channelId).filter(Boolean);

                if (!videoIds.length) continue;

                // Get video stats (videos.list = 1 unit)
                const videoStatsData = await ytGet('/videos', {
                    id: videoIds.join(','), part: 'statistics,snippet',
                });

                // Get channel stats (channels.list = 1 unit)
                const channelStatsData = await ytGet('/channels', {
                    id: [...new Set<string>(channelIds)].join(','), part: 'statistics,snippet',
                });

                const channelMap: Record<string, { subs: number; title: string; thumbnail: string }> = {};
                for (const ch of channelStatsData.items || []) {
                    channelMap[ch.id] = {
                        subs: parseInt(ch.statistics?.subscriberCount || '0'),
                        title: ch.snippet?.title || '',
                        thumbnail: ch.snippet?.thumbnails?.default?.url || '',
                    };
                }

                for (const video of videoStatsData.items || []) {
                    const chId = video.snippet?.channelId;
                    const ch = channelMap[chId];
                    if (!ch) continue;

                    const subs = ch.subs;
                    const views = parseInt(video.statistics?.viewCount || '0');

                    if (subs <= 5000 && views >= 10000 && subs > 0) {
                        const ratio = Math.round(views / subs);
                        if (ratio >= 20) {
                            embryoCandidates.push({
                                title: video.snippet?.title || '',
                                channel: ch.title,
                                channelId: chId,
                                views, subs, ratio,
                                thumbnail: video.snippet?.thumbnails?.medium?.url || '',
                                videoId: video.id,
                            });
                        }
                    }
                }
            } catch (err: any) {
                // If all keys are quota-exhausted, surface a useful message
                if (err?.quotaExhausted) {
                    return NextResponse.json({
                        total: 0, clusters: [], raw: [],
                        quotaError: err.message,
                    });
                }
                console.warn(`[Embryos] Seed "${q}" failed:`, err?.message);
            }
        }

        embryoCandidates.sort((a, b) => b.ratio - a.ratio);
        const clusterInput = embryoCandidates.map(c => ({ title: c.title, channel: c.channel }));
        const clusters = clusterTitles(clusterInput, 2);
        const enrichedClusters = clusters.map(cl => ({
            ...cl,
            members: embryoCandidates.filter(e => cl.titles.includes(e.title)).slice(0, 5),
        }));

        return NextResponse.json({
            total: embryoCandidates.length,
            clusters: enrichedClusters,
            raw: embryoCandidates.slice(0, 20),
        });

    } catch (err: any) {
        console.error('[Embryos]', err);
        return NextResponse.json({ error: err.message }, { status: 500 });
    }
}
